package org.apache.taglibs.extended.script.samples;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.ListIterator;

/**
 * A service for operations on a collection of Person objects. In a real application this 
 * would be backed by a data access layer connected to some database.
 * 
 * @author stani
 *
 */
public class PersonService {
	private List<Person> inMemoryPersonDatabase;
	
	
	public PersonService() {
		inMemoryPersonDatabase = new ArrayList<Person>();
		loadDefaultDataset();
	}
	
	/**
	 * Returns the whole list of people. 
	 */
	public List<Person> getPeople() {
		return inMemoryPersonDatabase;
	}
	
	
	/**
	 * Returns the person with a guven id, or null if such a person does not exist.
	 * @param id
	 * @return
	 */
	public Person getPerson(int id) {
		for (Person p : inMemoryPersonDatabase) {
			if (p.getId() == id)
				return p;
		}
		return null;
	}
	
	/**
	 * delete a person entry with a given id
	 * @param id
	 */
	public void deletePerson(int id) {
		ListIterator<Person> iter = inMemoryPersonDatabase.listIterator();
		while(iter.hasNext()) {
			Person p = iter.next();
			if(p.getId() == id) {
				iter.remove();
			}
		}
	}
	
	/**
	 * Saves Person as a new entry or updates existing entry if already known.
	 * 
	 * @param person
	 */
	public void saveOrUpdatePerson(Person person) {
		
		// first check if we are saving a new entry
		if (person.getId() == 0) {
			person.setId(generateNextID());
			inMemoryPersonDatabase.add(person);
		}
		
		// otherwise find the entry and update it
		ListIterator<Person> iter = inMemoryPersonDatabase.listIterator();
		while (iter.hasNext()) {
			Person p = iter.next();
			if (p.getId() == person.getId()) {
				iter.set(person); // replace old entry with new one
			}
		}
		
	}
	
	/**
	 * Initializes the in-memory database with default values. Needed because we are not 
	 * using a real database here.
	 */
	private void loadDefaultDataset() {
		Calendar c = new GregorianCalendar();		
		c.set(1983, 11, 1);	
		inMemoryPersonDatabase.add(new Person(1, "Grimmas", "Pendragon", c));
		
		c = new GregorianCalendar();
		c.set(1970, 5, 23);
		inMemoryPersonDatabase.add(new Person(2, "Max", "Mustermann", c));
		
		c = new GregorianCalendar();
		c.set(1965, 0, 20);
		inMemoryPersonDatabase.add(new Person(3, "Mana", "Musterfrau", c));
		
	}
	
	/**
	 * Computes the next id to be used. Must be higher than any current id. 
	 * @return
	 */
	private int generateNextID() {
		int newID = 0;
		for (Person p : inMemoryPersonDatabase) {
			if (p.getId() > newID)
				newID = p.getId();
		}
		return newID + 1;
	}
}
